#!/usr/bin/env python
"""
Exports dates and tasks from Open XChange
=========================================

One of the options -c or -t must be given so either dates or tasks
can be exported. The calendar folder id may be 0 so all dates will
be exported. 

Usage:

-c id, --calendar=id             The id of the calendar folder to
                                 export
-t id, --tasks=id                The id of the task folder to export

-s timestamp, --start=timestamp  Start date for the export
-e days, --end=days              Exports a range of `days` after the
                                 start date
-e timestamp, --end=timestamp    Specify the end timestamp of the
                                 export range
-x tag, --tag=tag                Export only dates/tasks with the
                                 specified tag
                                 
-a file, --template=file         Specify a Jinja2 template to use
                                 for generation of the output
-p path, --path=path             The path under which the export is
                                 stored
                                 
-h, --help                       Show this usage information
"""


#================================================
#                    SETTINGS
#================================================

USERNAME = ''
PASSWORD = ''
SERVER   = 'mail.piratenpartei-nrw.de'
SUBPATH  = '/'

#================================================


from httplib import HTTPSConnection
from urllib import urlencode
from datetime import date, datetime, timedelta
import json, sys, os, locale, re, getopt
from shutil import Error, WindowsError, copy2, copystat
from jinja2 import Environment, FileSystemLoader

template_path = os.path.dirname(os.getcwd() + '/' + __file__) + '/templates'
data_path     = os.path.dirname(os.getcwd() + '/' + __file__) + '/data'

env = Environment(loader=FileSystemLoader(template_path))

locale.setlocale(locale.LC_ALL, ('de_DE', 'UTF-8'))

def copytree(src, dst, symlinks=False, ignore=None):
    names = os.listdir(src)
    if ignore is not None:
        ignored_names = ignore(src, names)
    else:
        ignored_names = set()

    try:
        os.makedirs(dst)
    except OSError:
        pass
        
    errors = []
    for name in names:
        if name in ignored_names:
            continue
        srcname = os.path.join(src, name)
        dstname = os.path.join(dst, name)
        try:
            if symlinks and os.path.islink(srcname):
                linkto = os.readlink(srcname)
                os.symlink(linkto, dstname)
            elif os.path.isdir(srcname):
                copytree(srcname, dstname, symlinks, ignore)
            else:
                copy2(srcname, dstname)
            # XXX What about devices, sockets etc.?
        except (IOError, os.error), why:
            errors.append((srcname, dstname, str(why)))
        # catch the Error from the recursive copytree so that we can
        # continue with other files
        except Error, err:
            errors.extend(err.args[0])
    try:
        copystat(src, dst)
    except WindowsError:
        # can't copy file access times on Windows
        pass
    except OSError, why:
        errors.extend((src, dst, str(why)))
    if errors:
        raise Error(errors)


class OXError(Exception):
    def __init__(self, response):
        if isinstance(response, str):
            self.category = None
            self.id       = None
            self.code     = None
            self.msg      = response
        else:
            self.category = response['category']
            self.id       = response['error_id']
            self.code     = response['code']
            self.msg      = response['error'] + ' ' + \
                            str(tuple(response['error_params']))

    def __str__(self):
        if self.code is not None:
            return 'OX Error %s: %s' % (self.code, self.msg)
        else:
            return 'OX Error: %s' % self.msg

class OXConnector(object):
    def __init__(self, server, subpath='/'):
        self.server  = server
        self.subpath = subpath

        self.conn = HTTPSConnection(server)

        self.name     = None
        self.session  = None
        self.cookies  = {}

    def request(self, module, action, parameters={}, type='GET'):
        headers = {}

        if self.session is not None and not 'session' in parameters:
            parameters['session'] = self.session

        if len(self.cookies) > 0:
            cpairs = [x+'='+y for x,y in self.cookies.items()]
            headers['Cookie'] = ';'.join(cpairs)

        parameters['action'] = action

        rqstr = '%sajax/%s?%s' % (self.subpath, module, \
                                  urlencode(parameters))

        self.conn.request(type, rqstr, None, headers)

        response = self.conn.getresponse()

        cookies = response.getheader('Set-Cookie', '')

        if cookies != '':
            cookies = [map(lambda x: x.strip(), x.split(';')[0].split('=')) for x in cookies.split(',')]
            cookies = filter(lambda x: len(x) == 2, cookies)

            self.cookies.update(cookies)

        data = response.read()

        if data == '':
            return {}

        data = json.loads(data)

        if 'error' in data:
            raise OXError(data)

        return data
        
    def _convert_time(self, time):
        _time = float(time) / 1000
        
        return datetime.utcfromtimestamp(_time)
        
    def _convert_date(self, date):
        _date = float(date) / 1000
        
        return date.fromtimestamp(_date, Local)
        
    def _get_user_info(self, id):
        return self.request('user', 'get', {'id': str(id)})['data']
        
    def _get_group_info(self, id):
        return self.request('group', 'get', {'id': str(id)})['data']
        
    def _get_participant_info(self, data):
        if data['type'] == 1:
            return self._get_user_info(data['id'])
            
        if data['type'] == 2:
            return self._get_group_info(data['id'])
            
        return data
        

    def login(self, name, password):
        self.name = name
        
        params = {'name': name, 'password': password}

        response = self.request('login', 'login', params, 'POST')

        if 'session' in response:
            self.session = response['session']

        else:
            raise OXError('Unexspected error')

    def logout(self):
        if self.session is None:
            raise OXError('Cannot log out because not logged in')

        response = self.request('login', 'logout')

        if response:
            raise OXError(response)

        else:
            self.session = None
            self.name    = None

    def list_calendar(self, start=None, end=30, \
                      folder=None, columns=None):
                          
        SHOW_AS = [None, 'reserved', 'temporary', 'absent', 'free']
        PCPT_MAPPER = lambda y: self._get_participant_info(y)
    
        if start is None:
            start = date.today()
            
        if isinstance(end, int):
            end = start + timedelta(end)
            
        _start = start - date(1970, 1, 1)
        _start = '%d' % (_start.days * 8.64e7)
            
        _end = end - date(1970, 1, 1)
        _end = '%d' % (_end.days * 8.64e7)
        
        params = {'start': _start, 'end': _end}

        if folder is not None:
            params['folder'] = folder

        if columns is None:
            columns = [1,100,200,201,202,203,220,400,401,402]

        params['columns'] = ','.join(map(str, columns))
        
        data = self.request('calendar', 'all', params)['data']
        
        dates = []
        
        for entry in data:
            _entry = []
            
            for i, column in enumerate(columns):
                _entry.append({
                    1:   lambda x: ('id',           int(x)),
                    20:  lambda x: ('folder_id',    int(x)),
                    100: lambda x: ('tags',         x and map(lambda x: x.strip(), x.split(',')) or None),
                    200: lambda x: ('title',        x and unicode(x) or None),
                    201: lambda x: ('start',        x and self._convert_time(x) or None),
                    202: lambda x: ('end',          x and self._convert_time(x) or None),
                    203: lambda x: ('description',  x and unicode(x) or None),
                    220: lambda x: ('participants', map(PCPT_MAPPER, x)),
                    221: lambda x: ('users',        x),
                    400: lambda x: ('location',     x and unicode(x) or None),
                    401: lambda x: ('full_time',    bool(x)),
                    402: lambda x: ('shown_as',     SHOW_AS[int(x)])
                }[column](entry[i]))
                
            dates.append(dict(_entry))
        
        return dates

    def list_tasks(self, folder, start=None, end=30, columns=None):        
        PRIORITIES = [None, 'low', 'medium', 'high']
        STATI = [None, 'notstarted', 'inprogress', 'done', \
                 'waiting', 'deferred']
        STATI2 = [None, 0, 0, 2, 1, 1]
        PCPT_MAPPER = lambda y: self._get_participant_info(y)
    
        if start is None:
            start = date.today()
            
        if isinstance(end, int):
            end = start + timedelta(end)
            
        _start = start - date(1970, 1, 1)
        _start = '%d' % (_start.days * 8.64e7)
            
        _end = end - date(1970, 1, 1)
        _end = '%d' % (_end.days * 8.64e7)
        
        params = {'start': _start, 'end': _end}

        params['folder'] = folder

        if columns is None:
            columns = [1,100,200,201,202,203,220,300,301,309]

        params['columns'] = ','.join(map(str, columns))
        
        data = self.request('tasks', 'all', params)['data']
        
        tasks = []
        
        for entry in data:
            _entry = []
            
            for i, column in enumerate(columns):
                _tmp = {
                
                1:   lambda x: ('id',                int(x)),
                20:  lambda x: ('folder_id',         int(x)),
                100: lambda x: ('tags',              x and map(lambda x: x.strip(), x.split(',')) or None),
                200: lambda x: ('title',             x and unicode(x) or None),
                201: lambda x: ('start',             x and self._convert_time(x) or None),
                202: lambda x: ('end',               x and self._convert_time(x) or None),
                203: lambda x: ('description',       x and unicode(x) or None),
                220: lambda x: ('participants',      map(PCPT_MAPPER, x)),
                221: lambda x: ('users',             x),
                300: lambda x: [('status',           STATI[int(x)]), \
                                ('status_group',     STATI2[int(x)])],
                301: lambda x: ('percent_completed', int(x)),
                309: lambda x: ('priority',          PRIORITIES[int(x)])
                
                }[column](entry[i])
                
                if isinstance(_tmp, list):
                    _entry.extend(_tmp)
                else:
                    _entry.append(_tmp)
                                
            tasks.append(dict(_entry))
        
        return tasks
        
def _filename(string):
    return re.sub(r'[^A-Za-z0-9_\-]+', '-', string).lower()
    
env.filters['filename'] = _filename
        
def filter_by_tag(objects, tag):
    return filter(lambda x: x['tags'] and tag in x['tags'], objects)
        
def all_tags(objects):
    tags = set()
    
    for obj in objects:
        if obj['tags']:
            tags.update(obj['tags'])
    
    return list(tags)
    
def write_dates(oxc, path, template, folder=None, \
                tag=None, start=None, end=30):
                    
    try:
        copytree(data_path, path)
    except OSError:
        print "Couldn't copy all files to %s, " % path + \
              "maybe the data allready exists?"
              
    try:
        os.mkdir(path + 'tags')
    except OSError:
        print "Couldn't create %stags, " % path + \
              "maybe it allready exists?"
                    
    dates = oxc.list_calendar(folder=folder, start=start, end=end)
    dates = sorted(dates, lambda a,b: cmp(a['start'], b['start']))
    
    if start is None:
        start = date.today()
        
    if isinstance(end, int):
        end = start + timedelta(end)
    
    template = env.get_template(template)
    
    if tag is None:            
        tags  = {}
        
        for tag in all_tags(dates):
            tags[tag] = len(filter_by_tag(dates, tag))
            
        _min   = min(tags.values())
        _max   = max(tags.values())
        _range = _max - _min
        
        if _range == 0:
            _range = 1
        
        for tag in tags:        
            tags[tag] = (tags[tag] - _min)/float(_range)
        
        for tag in tags:
            _dates = filter_by_tag(dates, tag)
                  
            out = template.render(start=start, end=end, dates=_dates, \
                                  tags=tags, active_tag=tag, \
                                  root='../').encode('UTF-8')
            
            f = open(path+'tags/%s.html' % _filename(tag), 'w') 
            f.write(out)
            f.close()
        
        out = template.render(start=start, end=end, dates=dates, \
                              tags=tags).encode('UTF-8')
        
    else:
        dates = filter_by_tag(dates, tag)
        
        out = template.render(start=start, end=end, dates=dates, \
                              active_tag=tag).encode('UTF-8')
        
    f = open(path+'index.html', 'w') 
    f.write(out)
    f.close()
    
def write_tasks(oxc, path, template, folder, \
                tag=None, start=None, end=30):
                    
    PRIORITIES = {'low': 2, 'medium': 1, 'high': 0}
    
    try:
        copytree(data_path, path)
    except OSError:
        print "Couldn't copy all files to %s, " % path + \
              "maybe the data allready exists?"
              
    try:
        os.mkdir(path + 'tags')
    except OSError:
        print "Couldn't create %stags, " % path + \
              "maybe it allready exists?"
        
    
    tasks = oxc.list_tasks(folder=folder, start=start, end=end)
    tasks.sort(key=lambda x: (x['status_group'], \
                              x['start'] or datetime(1970, 1, 1), \
                              PRIORITIES[x['priority']]))
        
    if start is None:
        start = date.today()
        
    if isinstance(end, int):
        end = start + timedelta(end)
    
    template = env.get_template(template)
    
    if tag is None:            
        tags  = {}
        
        for tag in all_tags(tasks):
            tags[tag] = len(filter_by_tag(tasks, tag))
            
        _min   = min(tags.values())
        _max   = max(tags.values())
        _range = _max - _min
        
        if _range == 0:
            _range = 1
        
        for tag in tags:        
            tags[tag] = (tags[tag] - _min)/float(_range)
        
        for tag in tags:
            _tasks = filter_by_tag(tasks, tag)
                  
            out = template.render(start=start, end=end, tasks=_tasks, \
                                  tags=tags, active_tag=tag, \
                                  root='../').encode('UTF-8')
            
            f = open(path+'tags/%s.html' % _filename(tag), 'w') 
            f.write(out)
            f.close()
        
        out = template.render(start=start, end=end, tasks=tasks, \
                              tags=tags).encode('UTF-8')
        
    else:
        tasks = filter_by_tag(tasks, tag)
        
        out = template.render(start=start, end=end, tasks=tasks, \
                              active_tag=tag).encode('UTF-8')
        
    f = open(path+'index.html', 'w') 
    f.write(out)
    f.close()

def main():    
    try:
        longops = ["help", "path=", "tasks=", "calender=", "tag=", \
                   "template=", "start=", "end="]
        
        opts, args = getopt.getopt(sys.argv[1:], "hp:t:c:x:a:s:e:", \
                                   longops)
        
    except getopt.GetoptError, err:
        print str(err)
        print
        print __doc__
        sys.exit(2)
        
    oxc = OXConnector(SERVER, SUBPATH)

    oxc.login(USERNAME, PASSWORD)
        
    path     = ''
    tasks    = None
    calendar = None
    tag      = None
    template = None
    start    = None
    end      = 30
    
    for o, a in opts:
        if o in ("-h", "--help"):
            print main.__doc__
            sys.exit()
        elif o in ("-p", "--path"):
            path = a[-1] == '/' and a or a + '/'
        elif o in ("-t", "--tasks"):
            tasks = int(a)
        elif o in ("-c", "--calendar"):
            calendar = int(a)
        elif o in ("-x", "--tag"):
            tag = a
        elif o in ("-a", "--template"):
            template = a
        elif o in ("-s", "--start"):
            start = int(a)
        elif o in ("-e", "--end"):
            end = int(a)
        else:
            assert False, "unhandled option"
            
    if tasks and calendar:
        print 'Error: You can only export either tasks or the calendar'
        print
        print __doc__
        sys.exit(2)
            
    if tasks is None and calendar is None:
        print 'Error: You have to export either tasks or the calendar'
        print
        print __doc__
        sys.exit(2)
            
    if start:
        start = date.fromtimestamp(start)
        
    if end > 10000:
        end = date.fromtimestamp(end)
            
    if tasks:
        write_tasks(oxc, path, template or 'tasks.html', \
                    tasks, tag, start, end)
        
    if calendar is not None:          
        write_dates(oxc, path, template or 'calendar.html', \
                    calendar or None, tag, start, end)
        
    print 'Completed operation sucessfully'
    
    oxc.logout()

if __name__ == "__main__":
    try:
        main()

    except OXError as e:
        print str(e)
